<?php
/**
 * BricksHandler - Bricks Builder spezifische Behandlung
 * 
 * Behandelt Bricks-spezifische Probleme bei Backup/Restore:
 * - Korrekte Serialisierung/Deserialisierung
 * - URL-Migration in Bricks-Daten
 * - Reparatur von beschädigten Daten
 * - Global Classes, Templates, Optionen
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Compatibility;

if (!defined('ABSPATH')) {
    exit;
}

class BricksHandler {
    
    /** @var array Bricks Meta-Keys */
    private $bricks_meta_keys = [
        '_bricks_page_content_2',
        '_bricks_page_header_2',
        '_bricks_page_footer_2',
        '_bricks_page_settings',
        '_bricks_template_settings',
        '_bricks_template_type',
        '_bricks_editor_mode',
    ];
    
    /** @var array Bricks Options */
    private $bricks_options = [
        'bricks_global_classes',
        'bricks_global_elements',
        'bricks_global_settings',
        'bricks_color_palette',
        'bricks_color_palettes',
        'bricks_breakpoints',
        'bricks_remote_templates',
        'bricks_sidebars',
    ];
    
    /** @var array Bricks-Options die nur String-Ersetzung brauchen (keine Serialisierung) */
    private $bricks_string_options = [
        'bricks_font_face_rules',
    ];
    
    /** @var array Statistiken */
    private $stats = [];
    
    /**
     * Prüft ob Bricks aktiv ist
     * 
     * @return bool
     */
    public function isBricksActive(): bool {
        // Theme aktiv?
        $theme = wp_get_theme();
        if ($theme->get_template() === 'bricks' || $theme->get('Name') === 'Bricks') {
            return true;
        }
        
        // Bricks-Daten vorhanden?
        global $wpdb;
        $bricks_data = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key LIKE '_bricks_%'"
        );
        
        return $bricks_data > 0;
    }
    
    /**
     * Migriert URLs in Bricks-Daten
     * 
     * @param string $old_url Alte URL
     * @param string $new_url Neue URL
     * @return array Migrations-Ergebnis
     */
    public function migrateUrls(string $old_url, string $new_url): array {
        
        $this->stats = [
            'postmeta_updated' => 0,
            'options_updated' => 0,
            'errors' => [],
        ];
        
        // URL-Varianten
        $old_http = str_replace('https://', 'http://', $old_url);
        $old_https = str_replace('http://', 'https://', $old_url);
        
        // Postmeta migrieren
        $this->migratePostmeta($old_http, $new_url);
        $this->migratePostmeta($old_https, $new_url);
        
        // Options migrieren (serialisierte Daten)
        $this->migrateOptions($old_http, $new_url);
        $this->migrateOptions($old_https, $new_url);
        
        // String-Options migrieren (einfache String-Ersetzung)
        $this->migrateStringOptions($old_http, $new_url);
        $this->migrateStringOptions($old_https, $new_url);
        
        
        return $this->stats;
    }
    
    /**
     * Migriert URLs in Bricks Postmeta
     * 
     * KRITISCH: Diese Methode muss auch mit korrupten serialisierten Daten umgehen können!
     * Local by Flywheel's URL-Fixer zerstört die Serialisierung.
     */
    private function migratePostmeta(string $old_url, string $new_url): void {
        global $wpdb;
        
        foreach ($this->bricks_meta_keys as $meta_key) {
            // ALLE Einträge holen, nicht nur die mit der URL
            // Weil die Serialisierung korrupt sein könnte
            $entries = $wpdb->get_results($wpdb->prepare(
                "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s",
                $meta_key
            ), ARRAY_A);
            
            
            foreach ($entries as $entry) {
                $value = $entry['meta_value'];
                
                // Überspringe leere Werte
                if (empty($value)) {
                    continue;
                }
                
                // Prüfe ob die alte URL enthalten ist
                $has_url = strpos($value, $old_url) !== false;
                
                // Prüfe ob die Daten korrupt sind
                $is_corrupt = (strpos($value, 'a:') === 0 || strpos($value, 's:') === 0) 
                              && @unserialize($value) === false;
                
                // Wenn weder URL noch Korruption, überspringe
                if (!$has_url && !$is_corrupt) {
                    continue;
                }
                
                
                // Verwende die robuste Repair & Replace Methode
                $new_value = $this->forceRepairAndReplace($value, $old_url, $new_url);
                
                if ($new_value !== $value) {
                    $result = $wpdb->update(
                        $wpdb->postmeta,
                        ['meta_value' => $new_value],
                        ['post_id' => $entry['post_id'], 'meta_key' => $meta_key],
                        ['%s'],
                        ['%d', '%s']
                    );
                    
                    if ($result !== false) {
                        $this->stats['postmeta_updated']++;
                        
                    }
                }
            }
        }
    }
    
    /**
     * Robuste Reparatur und URL-Ersetzung
     * 
     * Arbeitet direkt auf dem serialisierten String:
     * 1. Ersetzt URLs
     * 2. Korrigiert String-Längenangaben
     * 3. Verifiziert das Ergebnis
     */
    private function forceRepairAndReplace(string $value, string $old_url, string $new_url): string {
        // Schritt 1: URL ersetzen (direkt im String)
        $replaced = str_replace($old_url, $new_url, $value);
        
        // Schritt 2: Längenangaben korrigieren
        $repaired = $this->fixSerializedLengths($replaced);
        
        // Schritt 3: Verifizieren
        $test = @unserialize($repaired);
        
        if ($test !== false) {
            // Erfolg! Neu serialisieren für garantiert korrekte Längen
            return serialize($test);
        }
        
        // Fallback: Aggressive Reparatur
        $aggressive = $this->aggressiveRepair($replaced);
        if ($aggressive !== null) {
            return serialize($aggressive);
        }
        
        // Letzte Möglichkeit: Original mit direkter Ersetzung
        return $this->directReplace($value, $old_url, $new_url);
    }
    
    /**
     * Korrigiert String-Längenangaben in serialisierten Daten
     */
    private function fixSerializedLengths(string $value): string {
        // Pattern: s:NUMBER:"STRING";
        return preg_replace_callback(
            '/s:(\d+):"((?:[^"\\\\]|\\\\.)*)";/s',
            function($matches) {
                $string = $matches[2];
                // Berechne korrekte Länge (Byte-Länge, nicht Zeichen)
                $correct_length = strlen($string);
                return 's:' . $correct_length . ':"' . $string . '";';
            },
            $value
        );
    }
    
    /**
     * Migriert URLs in Bricks Options mit verbesserter Reparatur
     */
    private function migrateOptions(string $old_url, string $new_url): void {
        global $wpdb;
        
        foreach ($this->bricks_options as $option_name) {
            $value = $wpdb->get_var($wpdb->prepare(
                "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
                $option_name
            ));
            
            if (!$value) {
                continue;
            }
            
            // Prüfe ob Option URL enthält
            $has_url = strpos($value, $old_url) !== false;
            
            // Versuche zu deserialisieren
            $data = @unserialize($value);
            
            if ($data !== false) {
                // Daten sind valide - ersetze URLs rekursiv
                if ($has_url) {
                    $data = $this->replaceUrlsRecursive($data, $old_url, $new_url);
                    $new_value = serialize($data);
                    
                    $wpdb->update(
                        $wpdb->options,
                        ['option_value' => $new_value],
                        ['option_name' => $option_name],
                        ['%s'],
                        ['%s']
                    );
                    
                    $this->stats['options_updated']++;
                    
                }
            } else {
                // Daten sind korrupt - aggressive Reparatur versuchen
                
                // Versuche aggressive Reparatur
                $data = $this->aggressiveRepair($value);
                
                if ($data !== null) {
                    // Repariert! Ersetze URLs und speichere
                    if ($has_url) {
                        $data = $this->replaceUrlsRecursive($data, $old_url, $new_url);
                    }
                    $new_value = serialize($data);
                    
                    $update_result = $wpdb->update(
                        $wpdb->options,
                        ['option_value' => $new_value],
                        ['option_name' => $option_name],
                        ['%s'],
                        ['%s']
                    );
                    
                    
                    // WICHTIG: Cache invalidieren damit WordPress den neuen Wert lädt
                    wp_cache_delete($option_name, 'options');
                    wp_cache_delete('alloptions', 'options');
                    
                    $this->stats['options_updated']++;
                    
                } else {
                    $this->stats['errors'][] = "Option $option_name konnte nicht repariert werden";
                }
            }
        }
    }
    
    /**
     * Rekursive URL-Ersetzung in Arrays/Objekten
     */
    private function replaceUrlsRecursive($data, string $old_url, string $new_url) {
        if (is_string($data)) {
            return str_replace($old_url, $new_url, $data);
        }
        
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->replaceUrlsRecursive($value, $old_url, $new_url);
            }
        }
        
        if (is_object($data)) {
            foreach ($data as $key => $value) {
                $data->$key = $this->replaceUrlsRecursive($value, $old_url, $new_url);
            }
        }
        
        return $data;
    }
    
    /**
     * Migriert URLs in String-Options (CSS, etc. - keine Serialisierung)
     */
    private function migrateStringOptions(string $old_url, string $new_url): void {
        global $wpdb;
        
        
        foreach ($this->bricks_string_options as $option_name) {
            $value = $wpdb->get_var($wpdb->prepare(
                "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
                $option_name
            ));
            
            
            if (!$value || strpos($value, $old_url) === false) {
                continue;
            }
            
            // Einfache String-Ersetzung
            $new_value = str_replace($old_url, $new_url, $value);
            
            if ($new_value !== $value) {
                $wpdb->update(
                    $wpdb->options,
                    ['option_value' => $new_value],
                    ['option_name' => $option_name],
                    ['%s'],
                    ['%s']
                );
                
                $this->stats['options_updated']++;
                
            }
        }
    }
    
    /**
     * Ersetzt URL in Bricks-Daten (serialisiert)
     */
    private function replaceUrlInBricksData(string $value, string $old_url, string $new_url): string {
        // Deserialisieren
        $data = @unserialize($value);
        
        if ($data === false) {
            // Versuche aggressive Reparatur
            $data = $this->aggressiveRepair($value);
            
            if ($data === null) {
                
                // Letzte Möglichkeit: Direkte Ersetzung mit Längenkorrektur
                return $this->directReplace($value, $old_url, $new_url);
            }
        }
        
        // Rekursiv ersetzen
        $data = $this->replaceUrlsRecursive($data, $old_url, $new_url);
        
        // Wieder serialisieren (WICHTIG: Bricks erwartet serialisierte Arrays!)
        return serialize($data);
    }
    
    /**
     * Repariert alle Bricks-Daten
     * 
     * @return array Reparatur-Ergebnis
     */
    public function repairAll(): array {
        
        $this->stats = [
            'postmeta_repaired' => 0,
            'options_repaired' => 0,
            'errors' => [],
            'invalid_entries' => [],
        ];
        
        // KRITISCH: Cache leeren VOR dem Reparieren
        wp_cache_flush();
        
        // Postmeta reparieren
        $this->repairPostmeta();
        
        // Options reparieren - FORCE MODE
        $this->repairOptionsForce();
        
        // KRITISCH: Cache erneut leeren NACH dem Reparieren
        wp_cache_flush();
        foreach ($this->bricks_options as $option_name) {
            wp_cache_delete($option_name, 'options');
        }
        wp_cache_delete('alloptions', 'options');
        
        
        return $this->stats;
    }
    
    /**
     * FORCE: Repariert Bricks Options und schreibt sie neu
     * Umgeht WordPress Caching durch direkten DB-Zugriff
     */
    private function repairOptionsForce(): void {
        global $wpdb;
        
        foreach ($this->bricks_options as $option_name) {
            // Direkt aus DB lesen (umgeht Cache)
            $value = $wpdb->get_var($wpdb->prepare(
                "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
                $option_name
            ));
            
            if (!$value) {
                continue;
            }
            
            // Versuche zu deserialisieren
            $data = @unserialize($value);
            
            if ($data === false && (strpos($value, 'a:') === 0 || strpos($value, 's:') === 0)) {
                
                // Aggressive Reparatur versuchen
                $data = $this->aggressiveRepair($value);
                
                if ($data !== null) {
                    // Neu serialisieren für korrekte Längen
                    $new_value = serialize($data);
                    
                    
                    // Direkt in DB schreiben
                    $wpdb->update(
                        $wpdb->options,
                        ['option_value' => $new_value],
                        ['option_name' => $option_name],
                        ['%s'],
                        ['%s']
                    );
                    
                    $this->stats['options_repaired']++;
                } else {
                    
                    $this->stats['errors'][] = "Option $option_name konnte nicht repariert werden";
                }
            } elseif ($data !== false) {
                // Daten sind OK, aber neu serialisieren um korrekte Längen sicherzustellen
                $new_value = serialize($data);
                
                if ($new_value !== $value) {
                    
                    $wpdb->update(
                        $wpdb->options,
                        ['option_value' => $new_value],
                        ['option_name' => $option_name],
                        ['%s'],
                        ['%s']
                    );
                    
                    $this->stats['options_repaired']++;
                }
            }
        }
    }
    
    /**
     * Repariert Bricks Postmeta
     */
    private function repairPostmeta(): void {
        global $wpdb;
        
        foreach ($this->bricks_meta_keys as $meta_key) {
            $entries = $wpdb->get_results($wpdb->prepare(
                "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s",
                $meta_key
            ), ARRAY_A);
            
            foreach ($entries as $entry) {
                $value = $entry['meta_value'];
                
                // Prüfen ob Reparatur nötig
                $test = @unserialize($value);
                
                if ($test === false && !empty($value)) {
                    
                    // Reparieren versuchen
                    $repaired = $this->repairSerialized($value);
                    $test = @unserialize($repaired);
                    
                    if ($test !== false) {
                        $wpdb->update(
                            $wpdb->postmeta,
                            ['meta_value' => $repaired],
                            ['post_id' => $entry['post_id'], 'meta_key' => $meta_key],
                            ['%s'],
                            ['%d', '%s']
                        );
                        
                        $this->stats['postmeta_repaired']++;
                    } else {
                        $this->stats['invalid_entries'][] = [
                            'post_id' => $entry['post_id'],
                            'meta_key' => $meta_key,
                            'preview' => substr($value, 0, 100),
                        ];
                    }
                }
            }
        }
    }
    
    /**
     * Repariert Bricks Options
     */
    private function repairOptions(): void {
        global $wpdb;
        
        foreach ($this->bricks_options as $option_name) {
            $value = $wpdb->get_var($wpdb->prepare(
                "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
                $option_name
            ));
            
            
            if (!$value) {
                continue;
            }
            
            // Prüfen ob Reparatur nötig
            $test = @unserialize($value);
            
            if ($test === false && strpos($value, 'a:') === 0) {
                
                // Reparieren versuchen
                $repaired = $this->repairSerialized($value);
                $test = @unserialize($repaired);
                
                if ($test !== false) {
                    $wpdb->update(
                        $wpdb->options,
                        ['option_value' => $repaired],
                        ['option_name' => $option_name],
                        ['%s'],
                        ['%s']
                    );
                    
                    $this->stats['options_repaired']++;
                } else {
                }
            }
        }
    }
    
    /**
     * Repariert beschädigte Serialisierung - AGGRESSIVER MODUS
     */
    private function repairSerialized(string $value): string {
        // WordPress %-Platzhalter zurücksetzen
        $value = preg_replace('/\{[a-f0-9]{64}\}/', '%', $value);
        
        // Längenangaben korrigieren - MULTI-PASS für verschachtelte Strukturen
        $max_passes = 5;
        $last_value = '';
        
        for ($pass = 0; $pass < $max_passes && $value !== $last_value; $pass++) {
            $last_value = $value;
            
            // Repariere String-Längen
            $value = preg_replace_callback(
                '/s:(\d+):"((?:[^"\\\\]|\\\\.)*)";/s',
                function($matches) {
                    $string = $matches[2];
                    // Länge korrekt berechnen (ohne Escaping-Overhead)
                    $actual_length = strlen($string);
                    return 's:' . $actual_length . ':"' . $string . '";';
                },
                $value
            );
        }
        
        return $value ?: $last_value;
    }
    
    /**
     * Aggressive Reparatur für komplex beschädigte Serialisierung
     * Versucht URLs direkt zu ersetzen und neu zu serialisieren
     */
    private function aggressiveRepair(string $value, string $old_url = '', string $new_url = ''): ?array {
        // Versuche verschiedene Reparatur-Strategien
        
        // Strategie 1: Einfache Reparatur
        $repaired = $this->repairSerialized($value);
        $result = @unserialize($repaired);
        if ($result !== false) {
            return $result;
        }
        
        // Strategie 2: Entferne potentielle BOM und unsichtbare Zeichen
        $clean = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $value);
        $clean = trim($clean);
        $repaired = $this->repairSerialized($clean);
        $result = @unserialize($repaired);
        if ($result !== false) {
            return $result;
        }
        
        // Strategie 3: JSON-Fallback (falls die Daten JSON sind)
        $json = @json_decode($value, true);
        if ($json !== null) {
            return $json;
        }
        
        // Strategie 4: Versuche, die korrupte Serialisierung zu "heilen"
        // Finde alle String-Definitionen und repariere sie einzeln
        $pattern = '/s:(\d+):"([^"]*)"/';
        $repaired = preg_replace_callback($pattern, function($m) {
            return 's:' . strlen($m[2]) . ':"' . $m[2] . '"';
        }, $value);
        
        // Versuche Deserialisierung
        $result = @unserialize($repaired);
        if ($result !== false) {
            return $result;
        }
        
        return null;
    }
    
    /**
     * Direkte Ersetzung mit Längenkorrektur
     */
    private function directReplace(string $value, string $old_url, string $new_url): string {
        $old_len = strlen($old_url);
        $new_len = strlen($new_url);
        
        // Einfache Ersetzung
        $result = str_replace($old_url, $new_url, $value);
        
        // Längen korrigieren
        return $this->repairSerialized($result);
    }
    
    /**
     * Prüft Bricks-Daten-Integrität
     * 
     * @return array Integritäts-Check
     */
    public function checkIntegrity(): array {
        global $wpdb;
        
        $result = [
            'total_entries' => 0,
            'valid' => 0,
            'invalid' => 0,
            'empty' => 0,
            'invalid_entries' => [],
        ];
        
        foreach ($this->bricks_meta_keys as $meta_key) {
            $entries = $wpdb->get_results($wpdb->prepare(
                "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s",
                $meta_key
            ), ARRAY_A);
            
            foreach ($entries as $entry) {
                $result['total_entries']++;
                
                if (empty($entry['meta_value'])) {
                    $result['empty']++;
                    continue;
                }
                
                $test = @unserialize($entry['meta_value']);
                
                if ($test === false && strpos($entry['meta_value'], 'a:') === 0) {
                    $result['invalid']++;
                    $result['invalid_entries'][] = [
                        'post_id' => $entry['post_id'],
                        'meta_key' => $meta_key,
                        'preview' => substr($entry['meta_value'], 0, 100),
                    ];
                } else {
                    $result['valid']++;
                }
            }
        }
        
        return $result;
    }
    
    /**
     * Getter für Statistiken
     */
    public function getStats(): array {
        return $this->stats;
    }
}

