<?php
namespace JBM;

if (!defined('ABSPATH')) {
    exit;
}

class Logger {
    
    private $log_file;
    
    /**
     * Eindeutiger Token für Log-Dateinamen (Sicherheit)
     * Wird einmalig generiert und in der DB gespeichert
     */
    private $log_token;
    
    public function __construct() {
        // Stelle sicher dass Backup-Verzeichnis existiert
        if (!file_exists(JBM_BACKUP_DIR)) {
            wp_mkdir_p(JBM_BACKUP_DIR);
        }
        
        // Sicherheits-Dateien erstellen/aktualisieren
        $this->create_security_files();
        
        // Sicherheit: Randomisierter Log-Dateiname
        $this->log_token = $this->get_log_token();
        $this->log_file = JBM_BACKUP_DIR . 'jbm-log-' . $this->log_token . '.log';
        
        // Migration: Alte Log-Datei umbenennen falls vorhanden
        $this->migrate_old_log_file();
    }
    
    /**
     * Erstellt Sicherheitsdateien für das Backup-Verzeichnis
     * Sicherheit: Schützt vor direktem Zugriff (Apache & Nginx)
     */
    private function create_security_files() {
        // .htaccess für Apache
        $htaccess = JBM_BACKUP_DIR . '.htaccess';
        $htaccess_content = "# Jenva Backup & Migration - Security\n";
        $htaccess_content .= "# Apache 2.4+\n";
        $htaccess_content .= "<IfModule mod_authz_core.c>\n";
        $htaccess_content .= "    <FilesMatch \"\\.(log|txt|json|sql)$\">\n";
        $htaccess_content .= "        Require all denied\n";
        $htaccess_content .= "    </FilesMatch>\n";
        $htaccess_content .= "    <FilesMatch \"\\.(zip|jbm)$\">\n";
        $htaccess_content .= "        Require all granted\n";
        $htaccess_content .= "    </FilesMatch>\n";
        $htaccess_content .= "</IfModule>\n\n";
        $htaccess_content .= "# Apache 2.2 (Legacy)\n";
        $htaccess_content .= "<IfModule !mod_authz_core.c>\n";
        $htaccess_content .= "    Order deny,allow\n";
        $htaccess_content .= "    Deny from all\n";
        $htaccess_content .= "    <FilesMatch \"\\.(zip|jbm)$\">\n";
        $htaccess_content .= "        Allow from all\n";
        $htaccess_content .= "    </FilesMatch>\n";
        $htaccess_content .= "</IfModule>\n\n";
        $htaccess_content .= "# Verhindere Directory Listing\n";
        $htaccess_content .= "Options -Indexes\n";
        
        // Immer aktualisieren für neueste Sicherheitsregeln
        @file_put_contents($htaccess, $htaccess_content);
        
        // index.php für zusätzlichen Schutz
        $index = JBM_BACKUP_DIR . 'index.php';
        if (!file_exists($index)) {
            @file_put_contents($index, "<?php // Silence is golden");
        }
        
        // index.html für Nginx (verhindert Directory Listing)
        $index_html = JBM_BACKUP_DIR . 'index.html';
        if (!file_exists($index_html)) {
            @file_put_contents($index_html, '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>403 Forbidden</title></head><body><h1>Forbidden</h1></body></html>');
        }
    }
    
    /**
     * Holt oder erstellt den Log-Token für randomisierten Dateinamen
     * Sicherheit: Verhindert dass Log-Dateien durch URL-Raten gefunden werden
     * 
     * @return string Der 32-stellige Token
     */
    private function get_log_token() {
        $token = get_option('jbm_log_token');
        
        if (empty($token)) {
            // Generiere einen kryptografisch sicheren Token
            if (function_exists('random_bytes')) {
                $token = bin2hex(random_bytes(16)); // 32 Zeichen
            } else {
                // Fallback für ältere PHP-Versionen
                $token = wp_generate_password(32, false);
            }
            
            update_option('jbm_log_token', $token, false); // autoload = false
        }
        
        return $token;
    }
    
    /**
     * Migriert alte Log-Datei zum neuen sicheren Format
     */
    private function migrate_old_log_file() {
        $old_log_file = JBM_BACKUP_DIR . 'backup-log.txt';
        
        if (file_exists($old_log_file) && !file_exists($this->log_file)) {
            // Alte Log-Datei umbenennen
            @rename($old_log_file, $this->log_file);
        } elseif (file_exists($old_log_file) && file_exists($this->log_file)) {
            // Beide existieren - alte Datei anhängen und löschen
            $old_content = @file_get_contents($old_log_file);
            if ($old_content) {
                @file_put_contents($this->log_file, $old_content, FILE_APPEND);
            }
            @unlink($old_log_file);
        }
    }
    
    /**
     * Schreibt eine Log-Nachricht
     * Übersetzt die Nachricht automatisch basierend auf bekannten Mustern
     * Sicherheit: Maskiert sensible Daten in den Log-Nachrichten
     */
    public function log($message, $level = 'info') {
        // Sicherheit: Sensible Daten in der Nachricht maskieren
        $message = $this->sanitize_log_message($message);
        
        // Übersetze die Nachricht, wenn WordPress geladen ist
        if (function_exists('__')) {
            $translated_message = $this->translate_log_message($message);
            $message = $translated_message;
        }
        
        $timestamp = wp_date('Y-m-d H:i:s');
        $log_entry = "[$timestamp] [$level] $message\n";
        
        // Verzeichnis nochmal prüfen vor dem Schreiben
        if (!file_exists(JBM_BACKUP_DIR)) {
            @wp_mkdir_p(JBM_BACKUP_DIR);
        }
        
        @file_put_contents($this->log_file, $log_entry, FILE_APPEND);
    }
    
    /**
     * Maskiert sensible Daten in Log-Nachrichten
     * Sicherheit: Schützt vor Information Disclosure in Log-Dateien
     * 
     * @param string $message Die zu bereinigende Nachricht
     * @return string Die bereinigte Nachricht
     */
    private function sanitize_log_message($message) {
        // Absolute Server-Pfade maskieren (z.B. /home/user/public_html/... -> .../wp-content/...)
        if (defined('ABSPATH')) {
            $abspath = ABSPATH;
            // Ersetze den vollständigen Server-Pfad durch einen relativen Marker
            $message = str_replace($abspath, '[WP_ROOT]/', $message);
        }
        
        // WP-Content-Pfad separat maskieren (falls nicht bereits durch ABSPATH ersetzt)
        if (defined('WP_CONTENT_DIR')) {
            $content_dir = WP_CONTENT_DIR;
            $message = str_replace($content_dir, '[WP_CONTENT]', $message);
        }
        
        // Datenbank-Credentials maskieren (falls versehentlich in Fehlermeldungen)
        if (defined('DB_PASSWORD') && !empty(DB_PASSWORD)) {
            $message = str_replace(DB_PASSWORD, '[DB_PASS_HIDDEN]', $message);
        }
        if (defined('DB_USER') && !empty(DB_USER)) {
            // Nur ersetzen wenn es aussieht wie ein Credential-Leak (nicht in normalem Text)
            $message = preg_replace('/password["\']?\s*[=:]\s*["\']?' . preg_quote(DB_USER, '/') . '/i', 'password=[HIDDEN]', $message);
        }
        
        // E-Mail-Adressen teilweise maskieren (user@domain.com -> u***@domain.com)
        $message = preg_replace_callback(
            '/([a-zA-Z0-9._%+-]+)@([a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/',
            function($matches) {
                $local = $matches[1];
                $domain = $matches[2];
                if (strlen($local) > 2) {
                    $local = substr($local, 0, 1) . str_repeat('*', min(3, strlen($local) - 1));
                }
                return $local . '@' . $domain;
            },
            $message
        );
        
        // Stack-Traces kürzen (nur erste 3 Zeilen behalten)
        if (preg_match('/Stack trace:.*$/s', $message, $matches)) {
            $trace_lines = explode("\n", $matches[0]);
            if (count($trace_lines) > 4) {
                $shortened_trace = implode("\n", array_slice($trace_lines, 0, 4)) . "\n[...truncated]";
                $message = str_replace($matches[0], $shortened_trace, $message);
            }
        }
        
        return $message;
    }
    
    /**
     * Übersetzt eine Log-Nachricht basierend auf bekannten Mustern
     */
    private function translate_log_message($message) {
        // Bekannte Muster und ihre Übersetzungen
        $patterns = [
            // Diagnose-Test
            '/🔧 DIAGNOSE-TEST GESTARTET/' => __('🔧 DIAGNOSE-TEST GESTARTET', 'jenva-backup-migration'),
            
            // Schedule-Klasse
            '/✅ Schedule-Klasse initialisiert - Cron-Hooks registriert/' => __('✅ Schedule-Klasse initialisiert - Cron-Hooks registriert', 'jenva-backup-migration'),
            '/✅ Schedule class initialized - Cron hooks registered/' => __('✅ Schedule-Klasse initialisiert - Cron-Hooks registriert', 'jenva-backup-migration'),
            
            // Backup erstellt
            '/Backup erfolgreich erstellt: (.+?) \(Dauer: (\d+)s\)/' => function($matches) {
                return sprintf(__('Backup erfolgreich erstellt: %s (Dauer: %ss)', 'jenva-backup-migration'), $matches[1], $matches[2]);
            },
            '/Backup successfully created: (.+?) \(Duration: (\d+)s\)/' => function($matches) {
                return sprintf(__('Backup erfolgreich erstellt: %s (Dauer: %ss)', 'jenva-backup-migration'), $matches[1], $matches[2]);
            },
            
            // Backup in Datenbank gespeichert
            '/Backup in Datenbank gespeichert: ID=(\d+)/' => function($matches) {
                return sprintf(__('Backup in Datenbank gespeichert (ID: %s)', 'jenva-backup-migration'), $matches[1]);
            },
            '/Backup in Datenbank gespeichert \(ID: (\d+)\)/' => function($matches) {
                return sprintf(__('Backup in Datenbank gespeichert (ID: %s)', 'jenva-backup-migration'), $matches[1]);
            },
            '/Backup saved in database \(ID: (\d+)\)/' => function($matches) {
                return sprintf(__('Backup in Datenbank gespeichert (ID: %s)', 'jenva-backup-migration'), $matches[1]);
            },
            '/Backup in Datenbank gespeichert/' => __('Backup in Datenbank gespeichert', 'jenva-backup-migration'),
            '/Backup saved in database/' => __('Backup in Datenbank gespeichert', 'jenva-backup-migration'),
            
            // Auto-Cleanup
            '/🧹 Auto-Cleanup gestartet: full=(\d+), db=(\d+), other=(\d+), exclude_id=(\d+)/' => function($matches) {
                return sprintf(__('🧹 Auto-Cleanup gestartet: full=%s, db=%s, other=%s, exclude_id=%s', 'jenva-backup-migration'), $matches[1], $matches[2], $matches[3], $matches[4]);
            },
            '/🧹 Auto-cleanup started: full=(\d+), db=(\d+), other=(\d+), exclude_id=(\d+)/' => function($matches) {
                return sprintf(__('🧹 Auto-Cleanup gestartet: full=%s, db=%s, other=%s, exclude_id=%s', 'jenva-backup-migration'), $matches[1], $matches[2], $matches[3], $matches[4]);
            },
            '/🧹 Auto-Cleanup abgeschlossen/' => __('🧹 Auto-Cleanup abgeschlossen', 'jenva-backup-migration'),
            '/🧹 Auto-cleanup completed/' => __('🧹 Auto-Cleanup abgeschlossen', 'jenva-backup-migration'),
            
            // Cleanup Details
            '/Keine Full-Backups zu löschen \((\d+) <= (\d+)\)/' => function($matches) {
                return sprintf(__('Keine Full-Backups zu löschen (%s <= %s)', 'jenva-backup-migration'), $matches[1], $matches[2]);
            },
            '/Keine DB-Backups zu löschen \((\d+) <= (\d+)\)/' => function($matches) {
                return sprintf(__('Keine DB-Backups zu löschen (%s <= %s)', 'jenva-backup-migration'), $matches[1], $matches[2]);
            },
            '/Keine anderen Backups zu löschen \((\d+) <= (\d+)\)/' => function($matches) {
                return sprintf(__('Keine anderen Backups zu löschen (%s <= %s)', 'jenva-backup-migration'), $matches[1], $matches[2]);
            },
            '/Full-Backups gesamt: (\d+), löschbar: (\d+), Limit: (\d+)/' => function($matches) {
                return sprintf(__('Full-Backups gesamt: %s, löschbar: %s, Limit: %s', 'jenva-backup-migration'), $matches[1], $matches[2], $matches[3]);
            },
            '/DB-Backups gesamt: (\d+), löschbar: (\d+), Limit: (\d+)/' => function($matches) {
                return sprintf(__('DB-Backups gesamt: %s, löschbar: %s, Limit: %s', 'jenva-backup-migration'), $matches[1], $matches[2], $matches[3]);
            },
            '/Andere Backups gesamt: (\d+), löschbar: (\d+), Limit: (\d+)/' => function($matches) {
                return sprintf(__('Andere Backups gesamt: %s, löschbar: %s, Limit: %s', 'jenva-backup-migration'), $matches[1], $matches[2], $matches[3]);
            },
            
            // ZIP Archive
            '/ZIP erstellt und verifiziert: (.+)/' => function($matches) {
                return sprintf(__('ZIP erstellt und verifiziert: %s', 'jenva-backup-migration'), $matches[1]);
            },
            '/📦 ZIP-Archiv erstellt: (\d+) Dateien, (.+?) \(Kompression: Level (\d+)\)/' => function($matches) {
                return sprintf(__('📦 ZIP-Archiv erstellt: %s Dateien, %s (Kompression: Level %s)', 'jenva-backup-migration'), $matches[1], $matches[2], $matches[3]);
            },
            '/ZIP close\(\) result: (.+)/' => function($matches) {
                return sprintf(__('ZIP close() result: %s', 'jenva-backup-migration'), $matches[1]);
            },
            '/Schließe ZIP-Archiv mit (\d+) Dateien\.\.\./' => function($matches) {
                return sprintf(__('Schließe ZIP-Archiv mit %s Dateien...', 'jenva-backup-migration'), $matches[1]);
            },
            '/Starte Datei-Iteration für ZIP \(Batch-Größe: (\d+)\)/' => function($matches) {
                return sprintf(__('Starte Datei-Iteration für ZIP (Batch-Größe: %s)', 'jenva-backup-migration'), $matches[1]);
            },
            '/Kompression-Level: (\d+)/' => function($matches) {
                return sprintf(__('Kompression-Level: %s', 'jenva-backup-migration'), $matches[1]);
            },
            '/Erstelle ZIP: (.+)/' => function($matches) {
                return sprintf(__('Erstelle ZIP: %s', 'jenva-backup-migration'), $matches[1]);
            },
            '/Erstelle ZIP-Archiv mit optimierter Kompression\.\.\./' => __('Erstelle ZIP-Archiv mit optimierter Kompression...', 'jenva-backup-migration'),
            '/Dateien kopiert: (\d+) Dateien/' => function($matches) {
                return sprintf(__('Dateien kopiert: %s Dateien', 'jenva-backup-migration'), $matches[1]);
            },
            '/Root-Datei gesichert: (.+)/' => function($matches) {
                return sprintf(__('Root-Datei gesichert: %s', 'jenva-backup-migration'), $matches[1]);
            },
            
            // Download
            '/✅ Download erfolgreich: (.+?) \((.+?) in (.+?)\)/' => function($matches) {
                return sprintf(__('✅ Download erfolgreich: %s (%s in %s)', 'jenva-backup-migration'), $matches[1], $matches[2], $matches[3]);
            },
            
            // Datenbank-Export
            '/Datenbank-Export abgeschlossen: (\d+) Tabellen, (.+)/' => function($matches) {
                return sprintf(__('Datenbank-Export abgeschlossen: %s Tabellen, %s', 'jenva-backup-migration'), $matches[1], $matches[2]);
            },
            '/Tabelle (.+?) exportiert: (\d+) Zeilen/' => function($matches) {
                return sprintf(__('Tabelle %s exportiert: %s Zeilen', 'jenva-backup-migration'), $matches[1], $matches[2]);
            },
            '/📊 Exportiere Tabellen: (\d+)\.\.\./' => function($matches) {
                return sprintf(__('📊 Exportiere Tabellen: %s...', 'jenva-backup-migration'), $matches[1]);
            },
            '/Sichere alle Dateien\.\.\./' => __('Sichere alle Dateien...', 'jenva-backup-migration'),
            
            // Weitere häufige Nachrichten
            '/Räume temporäre Dateien auf\.\.\./' => __('Räume temporäre Dateien auf...', 'jenva-backup-migration'),
            '/Cleaning up temporary files\.\.\./' => __('Räume temporäre Dateien auf...', 'jenva-backup-migration'),
            '/Speichere Backup in Datenbank\.\.\./' => __('Speichere Backup in Datenbank...', 'jenva-backup-migration'),
            '/Saving backup in database\.\.\./' => __('Speichere Backup in Datenbank...', 'jenva-backup-migration'),
            '/Backup-Pfad: (.+)/' => function($matches) {
                return sprintf(__('Backup-Pfad: %s', 'jenva-backup-migration'), $matches[1]);
            },
            '/Backup path: (.+)/' => function($matches) {
                return sprintf(__('Backup-Pfad: %s', 'jenva-backup-migration'), $matches[1]);
            },
            '/Backup-Größe: (.+)/' => function($matches) {
                return sprintf(__('Backup-Größe: %s', 'jenva-backup-migration'), $matches[1]);
            },
            '/Backup size: (.+)/' => function($matches) {
                return sprintf(__('Backup-Größe: %s', 'jenva-backup-migration'), $matches[1]);
            },
            '/Backup-ID: (\d+)/' => function($matches) {
                return sprintf(__('Backup-ID: %s', 'jenva-backup-migration'), $matches[1]);
            },
            '/Backup ID: (\d+)/' => function($matches) {
                return sprintf(__('Backup-ID: %s', 'jenva-backup-migration'), $matches[1]);
            },
        ];
        
        // Prüfe jedes Muster
        foreach ($patterns as $pattern => $translation) {
            if (is_callable($translation)) {
                if (preg_match($pattern, $message, $matches)) {
                    return $translation($matches);
                }
            } elseif (preg_match($pattern, $message)) {
                return $translation;
            }
        }
        
        // Fallback: Versuche direkte Übersetzung
        $translated = __($message, 'jenva-backup-migration');
        // Wenn die Übersetzung gleich der ursprünglichen Nachricht ist, wurde sie nicht gefunden
        // In diesem Fall geben wir die ursprüngliche Nachricht zurück
        return ($translated === $message) ? $message : $translated;
    }
    
    /**
     * Holt die letzten Log-Einträge
     * Übersetzt die Logs beim Anzeigen, damit sie in der aktuellen Sprache angezeigt werden
     */
    public function get_logs($lines = 100) {
        if (!file_exists($this->log_file)) {
            return [];
        }
        
        $content = file_get_contents($this->log_file);
        $all_lines = explode("\n", $content);
        $reversed_lines = array_slice(array_reverse($all_lines), 0, $lines);
        
        // Übersetze jeden Log-Eintrag
        $translated_lines = [];
        foreach ($reversed_lines as $line) {
            if (empty(trim($line))) {
                $translated_lines[] = $line;
                continue;
            }
            
            // Extrahiere die Nachricht aus dem Log-Eintrag (Format: [timestamp] [level] message)
            if (preg_match('/^\[([^\]]+)\] \[([^\]]+)\] (.+)$/', $line, $matches)) {
                $timestamp = $matches[1];
                $level = $matches[2];
                $message = $matches[3];
                
                // Übersetze die Nachricht
                $translated_message = $this->translate_log_message($message);
                
                // Baue den Log-Eintrag wieder zusammen
                $translated_lines[] = "[$timestamp] [$level] $translated_message";
            } else {
                // Falls das Format nicht passt, versuche trotzdem zu übersetzen
                $translated_lines[] = $this->translate_log_message($line);
            }
        }
        
        return $translated_lines;
    }
    
    /**
     * Löscht das Log
     */
    public function clear_log() {
        if (file_exists($this->log_file)) {
            unlink($this->log_file);
        }
    }
}

