<?php
/**
 * DatabaseExporter - Sauberer Datenbank-Export
 * 
 * Exportiert die WordPress-Datenbank mit:
 * - Korrekter Zeichenkodierung (UTF-8)
 * - Transaktionssicherheit
 * - Chunk-basiertem Export für große Tabellen
 * - Optionaler Prefix-Normalisierung
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Backup;

if (!defined('ABSPATH')) {
    exit;
}

class DatabaseExporter {
    
    /** @var int Zeilen pro Batch */
    const ROWS_PER_BATCH = 1000;
    
    /** @var int Max. INSERT-Größe (1 MB) */
    const MAX_INSERT_SIZE = 1048576;
    
    /** @var array Tabellen-Informationen */
    private $tables = [];
    
    /** @var int Gesamtzeilen */
    private $total_rows = 0;
    
    /** @var string SQL-Output */
    private $sql = '';
    
    /** @var array Konfiguration */
    private $config;
    
    /**
     * Erstellt einen neuen DatabaseExporter
     * 
     * @param array $config Konfiguration
     */
    public function __construct(array $config = []) {
        $this->config = array_merge([
            'include_drop_tables' => true,
            'include_create_tables' => true,
            'extended_insert' => true,
            'normalize_prefix' => false, // Prefix auf 'wp_' normalisieren
            'exclude_tables' => [], // Tabellen ausschließen
            'exclude_table_data' => [], // Nur Struktur, keine Daten
        ], $config);
    }
    
    /**
     * Exportiert die komplette Datenbank
     * 
     * @return array Export-Ergebnis
     */
    public function export(): array {
        global $wpdb;
        
        $this->sql = '';
        $this->tables = [];
        $this->total_rows = 0;
        
        // Header
        $this->addHeader();
        
        // Alle Tabellen holen
        $all_tables = $wpdb->get_col("SHOW TABLES");
        
        foreach ($all_tables as $table) {
            // Ausgeschlossene Tabellen überspringen
            if (in_array($table, $this->config['exclude_tables'])) {
                continue;
            }
            
            // Tabelle exportieren
            $table_info = $this->exportTable($table);
            $this->tables[] = $table_info;
            $this->total_rows += $table_info['rows'];
        }
        
        // Footer
        $this->addFooter();
        
        return [
            'sql' => $this->sql,
            'tables' => $this->tables,
            'total_rows' => $this->total_rows,
            'size' => strlen($this->sql),
        ];
    }
    
    /**
     * Exportiert eine einzelne Tabelle
     * 
     * @param string $table Tabellenname
     * @return array Tabellen-Info
     */
    private function exportTable(string $table): array {
        global $wpdb;
        
        $table_info = [
            'name' => $table,
            'rows' => 0,
            'size' => 0,
        ];
        
        // Tabellen-Kommentar
        $this->sql .= "\n-- --------------------------------------------------------\n";
        $this->sql .= "-- Table: `$table`\n";
        $this->sql .= "-- --------------------------------------------------------\n\n";
        
        // DROP TABLE
        if ($this->config['include_drop_tables']) {
            $this->sql .= "DROP TABLE IF EXISTS `$table`;\n\n";
        }
        
        // CREATE TABLE
        if ($this->config['include_create_tables']) {
            $create = $wpdb->get_row("SHOW CREATE TABLE `$table`", ARRAY_N);
            if ($create) {
                $create_sql = $create[1];
                
                // Prefix normalisieren wenn gewünscht
                if ($this->config['normalize_prefix'] && $wpdb->prefix !== 'wp_') {
                    $create_sql = str_replace("`{$wpdb->prefix}", "`wp_", $create_sql);
                }
                
                $this->sql .= $create_sql . ";\n\n";
            }
        }
        
        // Daten exportieren (wenn nicht nur Struktur)
        if (!in_array($table, $this->config['exclude_table_data'])) {
            $table_info['rows'] = $this->exportTableData($table);
        }
        
        return $table_info;
    }
    
    /**
     * Exportiert Tabellen-Daten
     * 
     * @param string $table Tabellenname
     * @return int Anzahl exportierter Zeilen
     */
    private function exportTableData(string $table): int {
        global $wpdb;
        
        // Zeilen zählen
        $row_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM `$table`");
        
        if ($row_count === 0) {
            return 0;
        }
        
        // Spalten-Info holen
        $columns = $wpdb->get_results("SHOW COLUMNS FROM `$table`", ARRAY_A);
        $column_names = array_column($columns, 'Field');
        
        // Prefix für INSERT
        $insert_table = $table;
        if ($this->config['normalize_prefix'] && $wpdb->prefix !== 'wp_') {
            $insert_table = str_replace($wpdb->prefix, 'wp_', $table);
        }
        
        $insert_prefix = "INSERT INTO `$insert_table` (`" . implode('`, `', $column_names) . "`) VALUES\n";
        
        // In Batches exportieren
        $offset = 0;
        $exported = 0;
        
        while ($offset < $row_count) {
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM `$table` LIMIT %d OFFSET %d",
                    self::ROWS_PER_BATCH,
                    $offset
                ),
                ARRAY_A
            );
            
            if (empty($rows)) {
                break;
            }
            
            if ($this->config['extended_insert']) {
                // Extended INSERT (mehrere Zeilen pro Statement)
                $this->exportRowsExtended($insert_prefix, $rows, $columns);
            } else {
                // Einzelne INSERTs
                $this->exportRowsSingle($insert_table, $column_names, $rows, $columns);
            }
            
            $exported += count($rows);
            $offset += self::ROWS_PER_BATCH;
            
            // Speicher freigeben
            unset($rows);
            if (function_exists('gc_collect_cycles')) {
                gc_collect_cycles();
            }
        }
        
        $this->sql .= "\n";
        
        return $exported;
    }
    
    /**
     * Exportiert Zeilen als Extended INSERT
     */
    private function exportRowsExtended(string $prefix, array $rows, array $columns): void {
        $values = [];
        $current_size = strlen($prefix);
        
        foreach ($rows as $row) {
            $row_values = $this->formatRowValues($row, $columns);
            $row_sql = '(' . implode(', ', $row_values) . ')';
            $row_size = strlen($row_sql);
            
            // Neues Statement wenn zu groß
            if ($current_size + $row_size > self::MAX_INSERT_SIZE && !empty($values)) {
                $this->sql .= $prefix . implode(",\n", $values) . ";\n";
                $values = [];
                $current_size = strlen($prefix);
            }
            
            $values[] = $row_sql;
            $current_size += $row_size + 2; // +2 für ",\n"
        }
        
        // Restliche Werte
        if (!empty($values)) {
            $this->sql .= $prefix . implode(",\n", $values) . ";\n";
        }
    }
    
    /**
     * Exportiert Zeilen als einzelne INSERTs
     */
    private function exportRowsSingle(string $table, array $columns, array $rows, array $column_info): void {
        foreach ($rows as $row) {
            $values = $this->formatRowValues($row, $column_info);
            $this->sql .= "INSERT INTO `$table` (`" . implode('`, `', $columns) . "`) VALUES (";
            $this->sql .= implode(', ', $values);
            $this->sql .= ");\n";
        }
    }
    
    /**
     * Formatiert Zeilenwerte für SQL
     */
    private function formatRowValues(array $row, array $columns): array {
        global $wpdb;
        
        $values = [];
        
        foreach ($columns as $column) {
            $field = $column['Field'];
            $type = strtolower($column['Type']);
            $value = $row[$field] ?? null;
            
            if ($value === null) {
                $values[] = 'NULL';
            } elseif ($this->isNumericType($type) && is_numeric($value)) {
                $values[] = $value;
            } elseif ($this->isBinaryType($type)) {
                // Binary-Daten als Hex
                $values[] = "0x" . bin2hex($value);
            } else {
                // String escapen
                $values[] = "'" . $wpdb->_real_escape($value) . "'";
            }
        }
        
        return $values;
    }
    
    /**
     * Prüft ob Typ numerisch ist
     */
    private function isNumericType(string $type): bool {
        $numeric_types = ['int', 'tinyint', 'smallint', 'mediumint', 'bigint', 'float', 'double', 'decimal'];
        
        foreach ($numeric_types as $num_type) {
            if (strpos($type, $num_type) === 0) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Prüft ob Typ binär ist
     */
    private function isBinaryType(string $type): bool {
        return strpos($type, 'blob') !== false || strpos($type, 'binary') !== false;
    }
    
    /**
     * Fügt SQL-Header hinzu
     */
    private function addHeader(): void {
        global $wpdb;
        
        $this->sql .= "-- JenvaBackupMigration Database Export\n";
        $this->sql .= "-- Version: 2.0.0\n";
        $this->sql .= "-- Generated: " . gmdate('Y-m-d H:i:s') . " UTC\n";
        $this->sql .= "-- WordPress: " . get_bloginfo('version') . "\n";
        $this->sql .= "-- Site URL: " . get_site_url() . "\n";
        $this->sql .= "-- Table Prefix: {$wpdb->prefix}\n";
        $this->sql .= "-- --------------------------------------------------------\n\n";
        
        // Einstellungen für Import
        $this->sql .= "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n";
        $this->sql .= "SET AUTOCOMMIT = 0;\n";
        $this->sql .= "START TRANSACTION;\n";
        $this->sql .= "SET time_zone = \"+00:00\";\n\n";
        
        // Charset
        $charset = $wpdb->charset ?: 'utf8mb4';
        $collate = $wpdb->collate ?: 'utf8mb4_unicode_ci';
        
        $this->sql .= "/*!40101 SET @OLD_CHARACTER_SET_CLIENT=@@CHARACTER_SET_CLIENT */;\n";
        $this->sql .= "/*!40101 SET @OLD_CHARACTER_SET_RESULTS=@@CHARACTER_SET_RESULTS */;\n";
        $this->sql .= "/*!40101 SET @OLD_COLLATION_CONNECTION=@@COLLATION_CONNECTION */;\n";
        $this->sql .= "/*!40101 SET NAMES {$charset} */;\n\n";
    }
    
    /**
     * Fügt SQL-Footer hinzu
     */
    private function addFooter(): void {
        $this->sql .= "\nCOMMIT;\n\n";
        $this->sql .= "/*!40101 SET CHARACTER_SET_CLIENT=@OLD_CHARACTER_SET_CLIENT */;\n";
        $this->sql .= "/*!40101 SET CHARACTER_SET_RESULTS=@OLD_CHARACTER_SET_RESULTS */;\n";
        $this->sql .= "/*!40101 SET COLLATION_CONNECTION=@OLD_COLLATION_CONNECTION */;\n";
        $this->sql .= "\n-- Export completed\n";
    }
    
    /**
     * Exportiert nur bestimmte Tabellen
     * 
     * @param array $tables Tabellennamen
     * @return array
     */
    public function exportTables(array $tables): array {
        global $wpdb;
        
        $this->sql = '';
        $this->tables = [];
        $this->total_rows = 0;
        
        $this->addHeader();
        
        foreach ($tables as $table) {
            // Prüfen ob Tabelle existiert
            $exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table));
            
            if ($exists) {
                $table_info = $this->exportTable($table);
                $this->tables[] = $table_info;
                $this->total_rows += $table_info['rows'];
            }
        }
        
        $this->addFooter();
        
        return [
            'sql' => $this->sql,
            'tables' => $this->tables,
            'total_rows' => $this->total_rows,
        ];
    }
}

