<?php
/**
 * UrlMigrator - Saubere URL-Migration
 * 
 * Ersetzt URLs in der Datenbank mit:
 * - Korrekter Behandlung von serialisierten Daten
 * - HTTP/HTTPS Varianten
 * - Keine Beschädigung von Serialisierung
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Restore;

if (!defined('ABSPATH')) {
    exit;
}

class UrlMigrator {
    
    /** @var array Statistiken */
    private $stats = [];
    
    /** @var callable|null Progress-Callback */
    private $progress_callback;
    
    /** @var array Bricks-Options die übersprungen werden sollen (BricksHandler ist zuständig) */
    private $skip_options = [
        'bricks_global_classes',
        'bricks_global_elements', 
        'bricks_global_settings',
        'bricks_color_palette',
        'bricks_color_palettes',
        'bricks_breakpoints',
        'bricks_remote_templates',
        'bricks_element_categories',
        'bricks_code_signatures',
        'bricks_theme_styles',
        'bricks_sidebars',
        'bricks_font_face_rules',  // CSS-String, keine Serialisierung
    ];
    
    /** @var array Optionen die NIEMALS modifiziert werden dürfen (enthalten % Platzhalter oder sind strukturell kritisch) */
    private $never_modify_options = [
        'permalink_structure',    // Enthält %postname%, %category% etc.
        'category_base',
        'tag_base',
        'rewrite_rules',          // Wird nach Restore neu generiert
        'cron',                   // WP-Cron Zeitpläne
        'active_plugins',         // Plugin-Liste
        'recently_activated',
        'uninstall_plugins',
        'widget_block',           // Widget-Konfiguration
        'sidebars_widgets',
    ];
    
    /**
     * Setzt den Progress-Callback
     */
    public function setProgressCallback(callable $callback): void {
        $this->progress_callback = $callback;
    }
    
    /**
     * Migriert URLs von alt zu neu
     * 
     * @param string $old_url Alte URL
     * @param string $new_url Neue URL
     * @return array Migrations-Ergebnis
     */
    public function migrate(string $old_url, string $new_url): array {
        global $wpdb;
        
        // URLs normalisieren
        $old_url = untrailingslashit($old_url);
        $new_url = untrailingslashit($new_url);
        
        if ($old_url === $new_url) {
            return ['skipped' => true, 'reason' => 'URLs identisch'];
        }
        
        $this->stats = [
            'old_url' => $old_url,
            'new_url' => $new_url,
            'changes' => 0,
            'tables_processed' => 0,
            'serialized_fixed' => 0,
            'errors' => [],
        ];
        
        // HTTP/HTTPS Varianten
        $url_variants = $this->getUrlVariants($old_url, $new_url);
        
        // Tabellen mit URL-Daten
        $tables = [
            $wpdb->options => ['option_value'],
            $wpdb->posts => ['post_content', 'post_excerpt', 'guid', 'post_content_filtered'],
            $wpdb->postmeta => ['meta_value'],
            $wpdb->comments => ['comment_content', 'comment_author_url'],
            $wpdb->commentmeta => ['meta_value'],
            $wpdb->termmeta => ['meta_value'],
            $wpdb->usermeta => ['meta_value'],
        ];
        
        foreach ($tables as $table => $columns) {
            // Prüfen ob Tabelle existiert
            $exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table));
            if (!$exists) {
                continue;
            }
            
            foreach ($columns as $column) {
                $this->migrateColumn($table, $column, $url_variants);
            }
            
            $this->stats['tables_processed']++;
        }
        
        // Site-URL und Home-URL aktualisieren
        $this->updateCoreUrls($new_url);
        
        return $this->stats;
    }
    
    /**
     * Migriert URLs in einer Tabellenspalte
     */
    private function migrateColumn(string $table, string $column, array $url_variants): void {
        global $wpdb;
        
        foreach ($url_variants as $old => $new) {
            // Zeilen mit der alten URL finden
            $rows = $wpdb->get_results($wpdb->prepare(
                "SELECT * FROM `$table` WHERE `$column` LIKE %s",
                '%' . $wpdb->esc_like($old) . '%'
            ), ARRAY_A);
            
            if (empty($rows)) {
                continue;
            }
            
            // Primary Key ermitteln
            $pk = $this->getPrimaryKey($table);
            
            foreach ($rows as $row) {
                // Bricks-Options überspringen (werden vom BricksHandler behandelt)
                if ($table === $wpdb->options && isset($row['option_name']) && in_array($row['option_name'], $this->skip_options)) {
                    continue;
                }
                
                // Kritische Optionen NIEMALS modifizieren (enthalten % Platzhalter)
                if ($table === $wpdb->options && isset($row['option_name']) && in_array($row['option_name'], $this->never_modify_options)) {
                    continue;
                }
                
                // Bricks postmeta überspringen (wird vom BricksHandler behandelt)
                if ($table === $wpdb->postmeta && isset($row['meta_key']) && strpos($row['meta_key'], '_bricks_') === 0) {
                    continue;
                }
                
                $value = $row[$column];
                $new_value = $this->replaceInValue($value, $old, $new);
                
                if ($new_value !== $value) {
                    $result = $wpdb->update(
                        $table,
                        [$column => $new_value],
                        [$pk => $row[$pk]],
                        ['%s'],
                        $this->getColumnFormat($pk)
                    );
                    
                    if ($result !== false) {
                        $this->stats['changes']++;
                    }
                }
            }
        }
    }
    
    /**
     * Ersetzt URL in einem Wert (berücksichtigt Serialisierung)
     */
    private function replaceInValue(string $value, string $old_url, string $new_url): string {
        // Prüfen ob serialisiert
        if ($this->isSerialized($value)) {
            return $this->replaceInSerialized($value, $old_url, $new_url);
        }
        
        // Prüfen ob JSON
        $json_test = json_decode($value, true);
        if ($json_test !== null && json_last_error() === JSON_ERROR_NONE) {
            return $this->replaceInJson($value, $old_url, $new_url);
        }
        
        // Einfache String-Ersetzung
        return str_replace($old_url, $new_url, $value);
    }
    
    /**
     * Ersetzt URL in serialisierten Daten
     */
    private function replaceInSerialized(string $value, string $old_url, string $new_url): string {
        // Deserialisieren
        $data = @unserialize($value);
        
        if ($data === false) {
            // Reparieren versuchen
            $repaired = $this->repairSerialized($value);
            $data = @unserialize($repaired);
            
            if ($data === false) {
                // Fallback: Direkte String-Ersetzung mit Längenkorrektur
                return $this->replaceInSerializedDirect($value, $old_url, $new_url);
            }
            
            $this->stats['serialized_fixed']++;
        }
        
        // Rekursiv ersetzen
        $data = $this->replaceUrlsRecursive($data, $old_url, $new_url);
        
        // Wieder serialisieren
        return serialize($data);
    }
    
    /**
     * Direkte Ersetzung in serialisierten Daten mit Längenkorrektur
     */
    private function replaceInSerializedDirect(string $value, string $old_url, string $new_url): string {
        $old_len = strlen($old_url);
        $new_len = strlen($new_url);
        $diff = $new_len - $old_len;
        
        // Ersetze und korrigiere Längenangaben
        $result = $value;
        $offset = 0;
        
        while (($pos = strpos($result, $old_url, $offset)) !== false) {
            // Finde die Längenangabe vor dem String
            $search_start = max(0, $pos - 20);
            $prefix = substr($result, $search_start, $pos - $search_start);
            
            if (preg_match('/s:(\d+):"$/', $prefix, $matches)) {
                $old_length = (int) $matches[1];
                $new_length = $old_length + $diff;
                
                // Ersetze die Länge
                $length_pos = $search_start + strrpos($prefix, 's:' . $matches[1]);
                $result = substr_replace(
                    $result,
                    's:' . $new_length . ':"',
                    $length_pos,
                    strlen('s:' . $matches[1] . ':"')
                );
                
                // Position anpassen
                $pos += strlen('s:' . $new_length) - strlen('s:' . $matches[1]);
            }
            
            // URL ersetzen
            $result = substr_replace($result, $new_url, $pos, $old_len);
            $offset = $pos + $new_len;
        }
        
        return $result;
    }
    
    /**
     * Ersetzt URL in JSON-Daten
     */
    private function replaceInJson(string $value, string $old_url, string $new_url): string {
        $data = json_decode($value, true);
        $data = $this->replaceUrlsRecursive($data, $old_url, $new_url);
        return json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    }
    
    /**
     * Rekursive URL-Ersetzung in Arrays/Objekten
     */
    private function replaceUrlsRecursive($data, string $old_url, string $new_url) {
        if (is_string($data)) {
            return str_replace($old_url, $new_url, $data);
        }
        
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->replaceUrlsRecursive($value, $old_url, $new_url);
            }
        }
        
        if (is_object($data)) {
            foreach ($data as $key => $value) {
                $data->$key = $this->replaceUrlsRecursive($value, $old_url, $new_url);
            }
        }
        
        return $data;
    }
    
    /**
     * Repariert beschädigte Serialisierung
     */
    private function repairSerialized(string $value): string {
        // Korrigiere Längenangaben
        return preg_replace_callback(
            '/s:(\d+):"(.*?)";/s',
            function($matches) {
                $string = $matches[2];
                $correct_length = strlen($string);
                return 's:' . $correct_length . ':"' . $string . '";';
            },
            $value
        );
    }
    
    /**
     * Prüft ob ein Wert serialisiert ist
     */
    private function isSerialized(string $value): bool {
        if (empty($value)) {
            return false;
        }
        
        // Quick check
        if ($value === 'N;' || $value === 'b:0;' || $value === 'b:1;') {
            return true;
        }
        
        if (strlen($value) < 4) {
            return false;
        }
        
        // Beginnt mit a: (array), s: (string), O: (object), i: (integer), d: (double)
        return preg_match('/^[aOsidbN][:;]/', $value) === 1;
    }
    
    /**
     * Gibt URL-Varianten zurück (HTTP/HTTPS)
     */
    private function getUrlVariants(string $old_url, string $new_url): array {
        $variants = [];
        
        // Exakte Variante
        $variants[$old_url] = $new_url;
        
        // HTTP zu HTTPS Varianten
        $old_http = str_replace('https://', 'http://', $old_url);
        $old_https = str_replace('http://', 'https://', $old_url);
        $new_http = str_replace('https://', 'http://', $new_url);
        $new_https = str_replace('http://', 'https://', $new_url);
        
        // Beide Protokoll-Varianten auf die neue URL mappen
        if ($old_http !== $old_url) {
            $variants[$old_http] = $new_url;
        }
        if ($old_https !== $old_url) {
            $variants[$old_https] = $new_url;
        }
        
        return $variants;
    }
    
    /**
     * Aktualisiert WordPress Core URLs
     * 
     * WICHTIG: Verwendet direkte SQL-Queries statt update_option(),
     * da nach einem DB-Import das WordPress-Caching nicht zuverlässig funktioniert!
     */
    private function updateCoreUrls(string $new_url): void {
        global $wpdb;
        
        
        // Direkte SQL-Queries - umgeht WordPress-Cache vollständig
        $wpdb->update(
            $wpdb->options,
            ['option_value' => $new_url],
            ['option_name' => 'siteurl'],
            ['%s'],
            ['%s']
        );
        
        $wpdb->update(
            $wpdb->options,
            ['option_value' => $new_url],
            ['option_name' => 'home'],
            ['%s'],
            ['%s']
        );
        
        
        // WordPress-Cache leeren
        wp_cache_flush();
        
        // Auch den Object-Cache für diese Optionen leeren
        wp_cache_delete('siteurl', 'options');
        wp_cache_delete('home', 'options');
        wp_cache_delete('alloptions', 'options');
    }
    
    /**
     * Ermittelt den Primary Key einer Tabelle
     */
    private function getPrimaryKey(string $table): string {
        global $wpdb;
        
        $columns = $wpdb->get_results("SHOW COLUMNS FROM `$table`", ARRAY_A);
        
        foreach ($columns as $column) {
            if ($column['Key'] === 'PRI') {
                return $column['Field'];
            }
        }
        
        // Fallback
        return $columns[0]['Field'] ?? 'id';
    }
    
    /**
     * Gibt das Format für eine Spalte zurück
     */
    private function getColumnFormat(string $column): array {
        // Die meisten PKs sind Integer
        if (in_array($column, ['ID', 'id', 'option_id', 'meta_id', 'comment_ID', 'term_id', 'umeta_id'])) {
            return ['%d'];
        }
        return ['%s'];
    }
    
    /**
     * Getter für Statistiken
     */
    public function getStats(): array {
        return $this->stats;
    }
}

