<?php
/**
 * BackupOrchestrator - Koordiniert den gesamten Backup-Prozess
 * 
 * Der Orchestrator führt das Backup als Mehrphasen-Prozess durch:
 * 1. Vorbereitung (DB-Bereinigung, Analyse)
 * 2. Datenbank-Export
 * 3. Plugin-Inventar erstellen
 * 4. Dateien sammeln und segmentieren
 * 5. Container finalisieren
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Backup;

use JenvaBackupMigration\Core\Container;
use JenvaBackupMigration\Core\ChunkProcessor;

if (!defined('ABSPATH')) {
    exit;
}

class BackupOrchestrator {
    
    /** @var string Backup-ID */
    private $backup_id;
    
    /** @var string Backup-Typ */
    private $backup_type;
    
    /** @var Container */
    private $container;
    
    /** @var DatabaseCleaner */
    private $db_cleaner;
    
    /** @var DatabaseExporter */
    private $db_exporter;
    
    /** @var PluginInventory */
    private $plugin_inventory;
    
    /** @var FileCollector */
    private $file_collector;
    
    /** @var ChunkProcessor */
    private $chunk_processor;
    
    /** @var callable|null Progress-Callback */
    private $progress_callback;
    
    /** @var array Konfiguration */
    private $config;
    
    /** @var array Statistiken */
    private $stats = [
        'phase' => '',
        'total_phases' => 5,
        'current_phase' => 0,
        'files_processed' => 0,
        'bytes_processed' => 0,
        'errors' => [],
    ];
    
    /**
     * Erstellt einen neuen BackupOrchestrator
     * 
     * @param string $backup_type 'full', 'database', 'files', 'slim'
     * @param array $config Konfiguration
     */
    public function __construct(string $backup_type = 'full', array $config = []) {
        $this->backup_id = $this->generateBackupId();
        $this->backup_type = $backup_type;
        
        $this->config = array_merge([
            'clean_database' => true,
            'optimize_images' => false,
            'exclude_cache' => true,
            'exclude_logs' => true,
            'exclude_other_backups' => true,
            'include_premium_plugins' => true,
            'compression_level' => 9,
            'backup_dir' => WP_CONTENT_DIR . '/jenva-backup-migration/',
        ], $config);
        
        // Komponenten initialisieren
        $this->initializeComponents();
    }
    
    /**
     * Setzt den Progress-Callback
     * 
     * @param callable $callback function($phase, $progress, $message)
     */
    public function setProgressCallback(callable $callback): void {
        $this->progress_callback = $callback;
    }
    
    /**
     * Führt das komplette Backup durch
     * 
     * @return array Backup-Ergebnis
     */
    public function run(): array {
        try {
            
            $this->reportProgress('start', 0, 'Backup wird gestartet...');
            
            // Phase 1: Vorbereitung
            $this->phase1_prepare();
            
            // Phase 2: Datenbank
            if (in_array($this->backup_type, ['full', 'database', 'slim'])) {
                $this->phase2_database();
            }
            
            // Phase 3: Plugin-Inventar
            if (in_array($this->backup_type, ['full', 'slim'])) {
                $this->phase3_plugins();
            }
            
            // Phase 4: Dateien
            if (in_array($this->backup_type, ['full', 'files', 'slim'])) {
                $this->phase4_files();
            }
            
            // Phase 5: Finalisierung
            $result = $this->phase5_finalize();
            
            $this->reportProgress('complete', 100, 'Backup abgeschlossen!');
            
            
            return [
                'success' => true,
                'backup_id' => $this->backup_id,
                'path' => $result['path'],
                'jbm_path' => $result['jbm_path'], // Die .jbm Datei für Download
                'size' => $result['total_size'],
                'manifest' => $result['manifest'],
                'stats' => $this->stats,
            ];
            
        } catch (\Exception $e) {
            
            $this->stats['errors'][] = $e->getMessage();
            
            return [
                'success' => false,
                'backup_id' => $this->backup_id,
                'error' => $e->getMessage(),
                'stats' => $this->stats,
            ];
        }
    }
    
    /**
     * Phase 1: Vorbereitung
     */
    private function phase1_prepare(): void {
        $this->stats['phase'] = 'prepare';
        $this->stats['current_phase'] = 1;
        $this->reportProgress('prepare', 5, 'Bereite Backup vor...');
        
        // Container erstellen
        $container_path = $this->config['backup_dir'] . $this->backup_id . '/';
        $this->container = new Container($container_path, $this->backup_type);
        $this->container->create();
        
        // Datenbank bereinigen (optional)
        if ($this->config['clean_database'] && $this->backup_type !== 'files') {
            $this->reportProgress('prepare', 10, 'Bereinige Datenbank...');
            $cleanup_stats = $this->db_cleaner->clean();
            $this->stats['db_cleanup'] = $cleanup_stats;
        }
        
        $this->reportProgress('prepare', 15, 'Vorbereitung abgeschlossen');
    }
    
    /**
     * Phase 2: Datenbank-Export
     */
    private function phase2_database(): void {
        $this->stats['phase'] = 'database';
        $this->stats['current_phase'] = 2;
        $this->reportProgress('database', 20, 'Exportiere Datenbank...');
        
        // DB exportieren
        $export_result = $this->db_exporter->export();
        
        // Zum Container hinzufügen
        $this->container->addDatabase(
            $export_result['sql'],
            $export_result['tables'],
            $this->stats['db_cleanup'] ?? []
        );
        
        $this->stats['database'] = [
            'tables' => count($export_result['tables']),
            'rows' => $export_result['total_rows'],
            'size' => strlen($export_result['sql']),
        ];
        
        $this->reportProgress('database', 35, 'Datenbank exportiert: ' . count($export_result['tables']) . ' Tabellen');
    }
    
    /**
     * Phase 3: Plugin-Inventar
     */
    private function phase3_plugins(): void {
        $this->stats['phase'] = 'plugins';
        $this->stats['current_phase'] = 3;
        $this->reportProgress('plugins', 40, 'Analysiere Plugins...');
        
        // Plugin-Inventar erstellen
        $plugins = $this->plugin_inventory->collect();
        
        $wp_org_count = 0;
        $premium_count = 0;
        
        foreach ($plugins as $plugin) {
            $this->container->addPlugin($plugin);
            
            if ($plugin['source'] === 'wordpress.org') {
                $wp_org_count++;
            } else {
                $premium_count++;
                
                // Premium/Custom Plugins sichern
                if ($this->config['include_premium_plugins'] && $plugin['backed_up']) {
                    // Plugin-Dateien zum Container hinzufügen
                    $plugin_dir = WP_PLUGIN_DIR . '/' . dirname($plugin['slug']);
                    if (is_dir($plugin_dir)) {
                        $this->addDirectoryToContainer($plugin_dir, 'plugins', 'plugins/' . dirname($plugin['slug']));
                    }
                }
            }
        }
        
        $this->stats['plugins'] = [
            'total' => count($plugins),
            'wp_org' => $wp_org_count,
            'premium' => $premium_count,
        ];
        
        $this->reportProgress('plugins', 50, "Plugin-Inventar: {$wp_org_count} WP.org, {$premium_count} Premium");
    }
    
    /**
     * Phase 4: Dateien sammeln
     */
    private function phase4_files(): void {
        $this->stats['phase'] = 'files';
        $this->stats['current_phase'] = 4;
        $this->reportProgress('files', 55, 'Sammle Dateien...');
        
        // Dateien sammeln (mit Ausschlüssen)
        $files = $this->file_collector->collect($this->getExclusions());
        $total_files = count($files);
        
        $this->stats['files'] = [
            'total' => $total_files,
            'processed' => 0,
            'excluded' => $this->file_collector->getExcludedCount(),
            'excluded_size' => $this->file_collector->getExcludedSize(),
        ];
        
        // Dateien in Chunks verarbeiten
        $processed = 0;
        $this->chunk_processor->setProgressCallback(function($done, $total, $msg) use (&$processed, $total_files) {
            $processed = $done;
            $percent = 55 + (($done / $total) * 35);
            $this->reportProgress('files', $percent, "Dateien: $done von $total_files");
        });
        
        $result = $this->chunk_processor->processArray($files, function($file) {
            return $this->processFile($file);
        });
        
        $this->stats['files']['processed'] = $result['processed'];
        
        // Excluded-Stats zum Manifest hinzufügen
        for ($i = 0; $i < $this->file_collector->getExcludedCount(); $i++) {
            $this->container->getManifest()->addExcluded(0);
        }
        
        $this->reportProgress('files', 90, "Dateien verarbeitet: {$result['processed']} von $total_files");
    }
    
    /**
     * Phase 5: Finalisierung
     */
    private function phase5_finalize(): array {
        $this->stats['phase'] = 'finalize';
        $this->stats['current_phase'] = 5;
        $this->reportProgress('finalize', 92, 'Finalisiere Backup...');
        
        // Timeout erhöhen für große Backups
        @set_time_limit(600); // 10 Minuten
        @ini_set('max_execution_time', '600');
        
        
        // Container schließen
        $result = $this->container->close();
        
        // Als .jbm exportieren (ZIP-komprimierte Datei)
        $jbm_path = $this->config['backup_dir'] . $this->generateBackupName() . '.jbm';
        
        
        $this->container->export($jbm_path);
        
        // Prüfen ob .jbm erfolgreich erstellt wurde
        if (file_exists($jbm_path)) {
            $jbm_size = filesize($jbm_path);
        } else {
        }
        
        $result['jbm_path'] = $jbm_path;
        
        $this->reportProgress('finalize', 98, 'Backup finalisiert');
        
        return $result;
    }
    
    // ========================================
    // HELPER METHODEN
    // ========================================
    
    /**
     * Initialisiert alle Komponenten
     */
    private function initializeComponents(): void {
        $state_dir = $this->config['backup_dir'] . 'temp/';
        wp_mkdir_p($state_dir);
        
        $this->db_cleaner = new DatabaseCleaner();
        $this->db_exporter = new DatabaseExporter();
        $this->plugin_inventory = new PluginInventory();
        $this->file_collector = new FileCollector();
        $this->chunk_processor = new ChunkProcessor($this->backup_id, $state_dir);
    }
    
    /**
     * Generiert eine eindeutige Backup-ID
     */
    private function generateBackupId(): string {
        return 'jbm_' . date('Ymd_His') . '_' . wp_generate_password(8, false);
    }
    
    /**
     * Generiert den Backup-Namen
     */
    private function generateBackupName(): string {
        $site_name = sanitize_file_name(get_bloginfo('name'));
        $site_name = $site_name ?: 'wordpress';
        $date = date('Y-m-d_H-i-s');
        
        return "{$site_name}_JBM_{$this->backup_type}_{$date}";
    }
    
    /**
     * Gibt die Ausschluss-Regeln zurück
     */
    private function getExclusions(): array {
        $exclusions = [];
        
        if ($this->config['exclude_cache']) {
            $exclusions[] = '*/cache/*';
            $exclusions[] = '*/object-cache.php';
            $exclusions[] = '*/.cache/*';
        }
        
        if ($this->config['exclude_logs']) {
            $exclusions[] = '*.log';
            $exclusions[] = '*/logs/*';
            $exclusions[] = '*/debug.log';
        }
        
        if ($this->config['exclude_other_backups']) {
            // Andere Backup-Plugin-Verzeichnisse
            $exclusions[] = '*/updraftplus/*';
            $exclusions[] = '*/backwpup-*/*';
            $exclusions[] = '*/ai1wm-backups/*';
            $exclusions[] = '*/backups/*';
            $exclusions[] = '*/backup/*';
            $exclusions[] = '*/duplicator/*';
            $exclusions[] = '*/jenva-backup-migration/*';
        }
        
        // Slim-Modus: Nur Premium-Plugins, keine WP.org
        if ($this->backup_type === 'slim') {
            $exclusions[] = WP_PLUGIN_DIR . '/*'; // Wird durch Plugin-Inventar ersetzt
        }
        
        return $exclusions;
    }
    
    /**
     * Verarbeitet eine einzelne Datei
     */
    private function processFile(array $file): bool {
        $source = $file['path'];
        $relative = $file['relative'];
        $type = $file['type']; // 'uploads', 'themes', 'plugins', 'other'
        
        return $this->container->addFile($source, $type, $relative);
    }
    
    /**
     * Fügt ein Verzeichnis zum Container hinzu
     */
    private function addDirectoryToContainer(string $dir, string $type, string $prefix): void {
        $files = $this->file_collector->collectDirectory($dir);
        
        foreach ($files as $file) {
            $relative = $prefix . '/' . $file['relative'];
            $this->container->addFile($file['path'], $type, $relative);
        }
    }
    
    /**
     * Meldet Fortschritt
     */
    private function reportProgress(string $phase, float $percent, string $message): void {
        if ($this->progress_callback) {
            call_user_func($this->progress_callback, $phase, $percent, $message);
        }
    }
    
    /**
     * Getter für Backup-ID
     */
    public function getBackupId(): string {
        return $this->backup_id;
    }
    
    /**
     * Getter für Stats
     */
    public function getStats(): array {
        return $this->stats;
    }
}

