<?php
/**
 * BuilderHandler - Generischer Handler für alle Builder-Plugins
 * 
 * Unterstützt:
 * - Bricks Builder
 * - Elementor
 * - Beaver Builder
 * - Divi Builder
 * - Gutenberg Blocks (Custom)
 * - Und andere Builder mit serialisierten Meta-Daten
 * 
 * Behandelt Builder-spezifische Probleme bei Backup/Restore:
 * - Korrekte Serialisierung/Deserialisierung
 * - URL-Migration in Builder-Daten
 * - Reparatur von beschädigten Daten
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Compatibility;

if (!defined('ABSPATH')) {
    exit;
}

class BuilderHandler {
    
    /** @var array Builder-spezifische Meta-Key-Patterns */
    private $builder_patterns = [
        // Bricks Builder
        '/^_bricks_/',
        
        // Elementor
        '/^_elementor_/',
        '/^elementor_/',
        
        // Beaver Builder
        '/^_fl_builder_/',
        '/^fl_builder_/',
        
        // Divi Builder
        '/^_et_pb_/',
        '/^et_pb_/',
        '/^divi_/',
        
        // Visual Composer / WPBakery
        '/^_wpb_/',
        '/^vc_/',
        
        // Oxygen Builder
        '/^ct_/',
        '/^ct_builder_/',
        
        // Thrive Architect / Thrive Theme Builder
        '/^tve_/',
        '/^thrive_/',
        
        // Gutenberg Blocks (Custom Meta)
        '/^_block_/',
        '/^_wp_block_/',
        
        // GeneratePress / GenerateBlocks
        '/^_generate_/',
        '/^generate_/',
        
        // Kadence Blocks
        '/^_kadence_/',
        '/^kadence_/',
        
        // Astra / Spectra
        '/^_astra_/',
        '/^astra_/',
        '/^spectra_/',
        
        // Neve / Otter Blocks
        '/^_neve_/',
        '/^neve_/',
        '/^otter_/',
    ];
    
    /** @var array Builder-spezifische Options-Patterns */
    private $builder_option_patterns = [
        // Bricks
        '/^bricks_/',
        
        // Elementor
        '/^elementor_/',
        
        // Beaver Builder
        '/^fl_builder_/',
        
        // Divi
        '/^et_/',
        '/^divi_/',
        
        // Visual Composer
        '/^wpb_/',
        '/^vc_/',
        
        // Oxygen
        '/^ct_/',
        
        // Thrive
        '/^tve_/',
        '/^thrive_/',
    ];
    
    /** @var array Statistiken */
    private $stats = [];
    
    /**
     * Prüft ob Builder-Daten vorhanden sind
     * 
     * @return bool
     */
    public function hasBuilderData(): bool {
        // Prüfe ob Builder Meta-Keys existieren
        $builder_meta_keys = $this->getBuilderMetaKeys();
        if (!empty($builder_meta_keys)) {
            return true;
        }
        
        // Prüfe ob Builder Options existieren
        $builder_options = $this->getBuilderOptions();
        if (!empty($builder_options)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Migriert URLs in allen Builder-Daten
     * 
     * @param string $old_url Alte URL
     * @param string $new_url Neue URL
     * @return array Migrations-Ergebnis
     */
    public function migrateUrls(string $old_url, string $new_url): array {
        $this->stats = [
            'postmeta_updated' => 0,
            'options_updated' => 0,
            'errors' => [],
        ];
        
        // URL-Varianten (HTTP/HTTPS)
        $url_variants = $this->getUrlVariants($old_url, $new_url);
        
        // Postmeta migrieren
        $this->migrateBuilderPostmeta($url_variants);
        
        // Options migrieren
        $this->migrateBuilderOptions($url_variants);
        
        return $this->stats;
    }
    
    /**
     * Migriert URLs in Builder Postmeta
     */
    private function migrateBuilderPostmeta(array $url_variants): void {
        global $wpdb;
        
        // Alle Builder Meta-Keys finden
        $builder_meta_keys = $this->getBuilderMetaKeys();
        
        foreach ($builder_meta_keys as $meta_key) {
            // Für jede URL-Variante
            foreach ($url_variants as $old_url => $new_url) {
                $entries = $wpdb->get_results($wpdb->prepare(
                    "SELECT post_id, meta_value FROM {$wpdb->postmeta} 
                     WHERE meta_key = %s AND meta_value LIKE %s",
                    $meta_key,
                    '%' . $wpdb->esc_like($old_url) . '%'
                ), ARRAY_A);
                
                foreach ($entries as $entry) {
                    $new_value = $this->replaceUrlInBuilderData($entry['meta_value'], $old_url, $new_url);
                    
                    if ($new_value !== $entry['meta_value']) {
                        $result = $wpdb->update(
                            $wpdb->postmeta,
                            ['meta_value' => $new_value],
                            ['post_id' => $entry['post_id'], 'meta_key' => $meta_key],
                            ['%s'],
                            ['%d', '%s']
                        );
                        
                        if ($result !== false) {
                            $this->stats['postmeta_updated']++;
                        } else {
                            $this->stats['errors'][] = "Fehler beim Update von post_id {$entry['post_id']}, meta_key {$meta_key}";
                        }
                    }
                }
            }
        }
    }
    
    /**
     * Migriert URLs in Builder Options
     */
    private function migrateBuilderOptions(array $url_variants): void {
        global $wpdb;
        
        // Alle Builder Options finden
        $builder_options = $this->getBuilderOptions();
        
        foreach ($builder_options as $option_name) {
            $value = $wpdb->get_var($wpdb->prepare(
                "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
                $option_name
            ));
            
            if (!$value) {
                continue;
            }
            
            // Für jede URL-Variante prüfen und ersetzen
            foreach ($url_variants as $old_url => $new_url) {
                if (strpos($value, $old_url) !== false) {
                    $new_value = $this->replaceUrlInBuilderData($value, $old_url, $new_url);
                    
                    if ($new_value !== $value) {
                        $result = $wpdb->update(
                            $wpdb->options,
                            ['option_value' => $new_value],
                            ['option_name' => $option_name],
                            ['%s'],
                            ['%s']
                        );
                        
                        if ($result !== false) {
                            $this->stats['options_updated']++;
                        } else {
                            $this->stats['errors'][] = "Fehler beim Update von option {$option_name}";
                        }
                        
                        $value = $new_value; // Für nächste Iteration
                    }
                }
            }
        }
    }
    
    /**
     * Ersetzt URL in Builder-Daten (serialisiert/JSON)
     */
    private function replaceUrlInBuilderData(string $value, string $old_url, string $new_url): string {
        // Prüfen ob serialisiert
        if ($this->isSerialized($value)) {
            return $this->replaceInSerialized($value, $old_url, $new_url);
        }
        
        // Prüfen ob JSON
        $json_test = json_decode($value, true);
        if ($json_test !== null && json_last_error() === JSON_ERROR_NONE) {
            return $this->replaceInJson($value, $old_url, $new_url);
        }
        
        // Einfache String-Ersetzung
        return str_replace($old_url, $new_url, $value);
    }
    
    /**
     * Ersetzt URL in serialisierten Daten
     */
    private function replaceInSerialized(string $value, string $old_url, string $new_url): string {
        // Deserialisieren
        $data = @unserialize($value);
        
        if ($data === false) {
            // Reparieren versuchen
            $repaired = $this->repairSerialized($value);
            $data = @unserialize($repaired);
            
            if ($data === false) {
                // Fallback: Direkte Ersetzung mit Längenkorrektur
                return $this->directReplaceWithLengthCorrection($value, $old_url, $new_url);
            }
        }
        
        // Rekursiv ersetzen
        $data = $this->replaceUrlsRecursive($data, $old_url, $new_url);
        
        // Wieder serialisieren (WICHTIG: Builder erwarten serialisierte Arrays!)
        return serialize($data);
    }
    
    /**
     * Ersetzt URL in JSON-Daten
     */
    private function replaceInJson(string $value, string $old_url, string $new_url): string {
        $data = json_decode($value, true);
        
        if ($data === null) {
            // Fallback: Direkte String-Ersetzung
            return str_replace($old_url, $new_url, $value);
        }
        
        $data = $this->replaceUrlsRecursive($data, $old_url, $new_url);
        return json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    }
    
    /**
     * Rekursive URL-Ersetzung in Arrays/Objekten
     */
    private function replaceUrlsRecursive($data, string $old_url, string $new_url) {
        if (is_string($data)) {
            return str_replace($old_url, $new_url, $data);
        }
        
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->replaceUrlsRecursive($value, $old_url, $new_url);
            }
        }
        
        if (is_object($data)) {
            foreach ($data as $key => $value) {
                $data->$key = $this->replaceUrlsRecursive($value, $old_url, $new_url);
            }
        }
        
        return $data;
    }
    
    /**
     * Repariert beschädigte Serialisierung
     */
    private function repairSerialized(string $value): string {
        // WordPress %-Platzhalter zurücksetzen (von $wpdb->prepare)
        $value = preg_replace('/\{[a-f0-9]{64}\}/', '%', $value);
        
        // Längenangaben korrigieren
        $repaired = preg_replace_callback(
            '/s:(\d+):"((?:[^"\\\\]|\\\\.)*)";/s',
            function($matches) {
                $string = $matches[2];
                // Backslashes korrekt zählen
                $length = strlen(stripslashes($string));
                return 's:' . $length . ':"' . $string . '";';
            },
            $value
        );
        
        return $repaired ?: $value;
    }
    
    /**
     * Direkte Ersetzung mit Längenkorrektur für serialisierte Daten
     */
    private function directReplaceWithLengthCorrection(string $value, string $old_url, string $new_url): string {
        $old_len = strlen($old_url);
        $new_len = strlen($new_url);
        $diff = $new_len - $old_len;
        
        // Ersetze URL
        $result = str_replace($old_url, $new_url, $value);
        
        // Längen korrigieren
        return $this->repairSerialized($result);
    }
    
    /**
     * Gibt URL-Varianten zurück (HTTP/HTTPS)
     */
    private function getUrlVariants(string $old_url, string $new_url): array {
        $variants = [];
        
        // Exakte Variante
        $variants[$old_url] = $new_url;
        
        // HTTP zu HTTPS Varianten
        $old_http = str_replace('https://', 'http://', $old_url);
        $old_https = str_replace('http://', 'https://', $old_url);
        
        if ($old_http !== $old_url) {
            $variants[$old_http] = $new_url;
        }
        if ($old_https !== $old_url) {
            $variants[$old_https] = $new_url;
        }
        
        return $variants;
    }
    
    /**
     * Findet alle Builder Meta-Keys in der Datenbank
     */
    private function getBuilderMetaKeys(): array {
        global $wpdb;
        
        $meta_keys = [];
        
        // Alle Meta-Keys abrufen, die Builder-Patterns entsprechen
        $all_meta_keys = $wpdb->get_col(
            "SELECT DISTINCT meta_key FROM {$wpdb->postmeta} WHERE meta_key != ''"
        );
        
        foreach ($all_meta_keys as $meta_key) {
            foreach ($this->builder_patterns as $pattern) {
                if (preg_match($pattern, $meta_key)) {
                    $meta_keys[$meta_key] = true;
                    break;
                }
            }
        }
        
        return array_keys($meta_keys);
    }
    
    /**
     * Findet alle Builder Options in der Datenbank
     */
    private function getBuilderOptions(): array {
        global $wpdb;
        
        $options = [];
        
        // Alle Options abrufen, die Builder-Patterns entsprechen
        $all_options = $wpdb->get_col(
            "SELECT option_name FROM {$wpdb->options} WHERE option_name != ''"
        );
        
        foreach ($all_options as $option_name) {
            foreach ($this->builder_option_patterns as $pattern) {
                if (preg_match($pattern, $option_name)) {
                    $options[$option_name] = true;
                    break;
                }
            }
        }
        
        return array_keys($options);
    }
    
    /**
     * Prüft ob ein Wert serialisiert ist
     */
    private function isSerialized(string $value): bool {
        if (empty($value)) {
            return false;
        }
        
        // Quick checks
        if ($value === 'N;' || $value === 'b:0;' || $value === 'b:1;') {
            return true;
        }
        
        if (strlen($value) < 4) {
            return false;
        }
        
        // Beginnt mit a: (array), s: (string), O: (object), i: (integer), d: (double)
        return preg_match('/^[aOsidbN][:;]/', $value) === 1;
    }
    
    /**
     * Repariert alle Builder-Daten
     * 
     * @return array Reparatur-Ergebnis
     */
    public function repairAll(): array {
        $this->stats = [
            'postmeta_repaired' => 0,
            'options_repaired' => 0,
            'errors' => [],
        ];
        
        // Postmeta reparieren
        $this->repairBuilderPostmeta();
        
        // Options reparieren
        $this->repairBuilderOptions();
        
        return $this->stats;
    }
    
    /**
     * Repariert Builder Postmeta
     */
    private function repairBuilderPostmeta(): void {
        global $wpdb;
        
        $builder_meta_keys = $this->getBuilderMetaKeys();
        
        foreach ($builder_meta_keys as $meta_key) {
            $entries = $wpdb->get_results($wpdb->prepare(
                "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s",
                $meta_key
            ), ARRAY_A);
            
            foreach ($entries as $entry) {
                $value = $entry['meta_value'];
                
                // Prüfen ob Reparatur nötig
                if ($this->isSerialized($value)) {
                    $test = @unserialize($value);
                    
                    if ($test === false) {
                        // Reparieren versuchen
                        $repaired = $this->repairSerialized($value);
                        $test = @unserialize($repaired);
                        
                        if ($test !== false) {
                            $wpdb->update(
                                $wpdb->postmeta,
                                ['meta_value' => $repaired],
                                ['post_id' => $entry['post_id'], 'meta_key' => $meta_key],
                                ['%s'],
                                ['%d', '%s']
                            );
                            
                            $this->stats['postmeta_repaired']++;
                        }
                    }
                }
            }
        }
    }
    
    /**
     * Repariert Builder Options
     */
    private function repairBuilderOptions(): void {
        global $wpdb;
        
        $builder_options = $this->getBuilderOptions();
        
        foreach ($builder_options as $option_name) {
            $value = $wpdb->get_var($wpdb->prepare(
                "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
                $option_name
            ));
            
            if (!$value) {
                continue;
            }
            
            // Prüfen ob Reparatur nötig
            if ($this->isSerialized($value)) {
                $test = @unserialize($value);
                
                if ($test === false) {
                    // Reparieren versuchen
                    $repaired = $this->repairSerialized($value);
                    $test = @unserialize($repaired);
                    
                    if ($test !== false) {
                        $wpdb->update(
                            $wpdb->options,
                            ['option_value' => $repaired],
                            ['option_name' => $option_name],
                            ['%s'],
                            ['%s']
                        );
                        
                        $this->stats['options_repaired']++;
                    }
                }
            }
        }
    }
    
    /**
     * Getter für Statistiken
     */
    public function getStats(): array {
        return $this->stats;
    }
}

