<?php
namespace JBM;

if (!defined('ABSPATH')) {
    exit;
}

class Database {
    
    /**
     * Holt alle WordPress-Tabellen
     * Sicherheit: Nur Tabellen mit WordPress-Prefix werden zurückgegeben
     */
    public function get_tables() {
        global $wpdb;
        
        $tables = [];
        $results = $wpdb->get_results('SHOW TABLES', ARRAY_N);
        
        foreach ($results as $result) {
            $table_name = $result[0];
            
            // Nur WordPress-Tabellen (Sicherheitsvalidierung)
            if (strpos($table_name, $wpdb->prefix) === 0) {
                // Zusätzliche Validierung: Nur alphanumerische Zeichen und Unterstriche erlaubt
                if ($this->is_valid_table_name($table_name)) {
                    $tables[] = $table_name;
                }
            }
        }
        
        return $tables;
    }
    
    /**
     * Validiert einen Tabellennamen auf sichere Zeichen
     * Verhindert SQL Injection durch Whitelist-Ansatz
     * 
     * @param string $table_name Der zu validierende Tabellenname
     * @return bool True wenn gültig, sonst False
     */
    private function is_valid_table_name($table_name) {
        // Nur alphanumerische Zeichen, Unterstriche und Bindestriche erlaubt
        // Muss mit Buchstabe oder Prefix beginnen
        return preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $table_name) === 1;
    }
    
    /**
     * Escaped einen Tabellennamen sicher für SQL-Queries
     * 
     * @param string $table_name Der Tabellenname
     * @return string|false Der escaped Tabellenname oder false bei ungültigem Namen
     */
    private function escape_table_name($table_name) {
        global $wpdb;
        
        // Validierung: Muss gültiger Tabellenname sein
        if (!$this->is_valid_table_name($table_name)) {
            return false;
        }
        
        // Muss mit WordPress-Prefix beginnen (Sicherheit)
        if (strpos($table_name, $wpdb->prefix) !== 0) {
            return false;
        }
        
        // Backticks escapen (falls jemand versucht sie einzuschleusen)
        return str_replace('`', '``', $table_name);
    }
    
    /**
     * Berechnet die Größe der Datenbank
     * Sicherheit: Verwendet sichere Tabellennamen-Validierung
     */
    public function get_database_size() {
        global $wpdb;
        
        $size = 0;
        $tables = $this->get_tables();
        
        foreach ($tables as $table) {
            // Sicherheit: Tabellenname validieren und escapen
            $safe_table = $this->escape_table_name($table);
            if ($safe_table === false) {
                continue; // Ungültigen Tabellennamen überspringen
            }
            
            // Prepared Statement für LIKE-Clause
            $result = $wpdb->get_row(
                $wpdb->prepare("SHOW TABLE STATUS LIKE %s", $table)
            );
            
            if ($result) {
                $size += $result->Data_length + $result->Index_length;
            }
        }
        
        return $size;
    }
    
    /**
     * Optimiert die Datenbank
     * Sicherheit: Verwendet sichere Tabellennamen-Validierung
     */
    public function optimize() {
        global $wpdb;
        
        $tables = $this->get_tables();
        
        foreach ($tables as $table) {
            // Sicherheit: Tabellenname validieren und escapen
            $safe_table = $this->escape_table_name($table);
            if ($safe_table === false) {
                continue; // Ungültigen Tabellennamen überspringen
            }
            
            // OPTIMIZE TABLE mit sicher escaptem Tabellennamen
            // Hinweis: Prepared Statements unterstützen keine Identifier (Tabellennamen)
            // Daher verwenden wir Whitelist-Validierung + Escaping
            $wpdb->query("OPTIMIZE TABLE `{$safe_table}`");
        }
        
        return true;
    }
}

